/** @file   mobileobject.cpp
 * @brief   Implementation of MobileObject class
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:44 $
 * @author  Tomi Lamminsaari
 */


#include "mobileobject.h" // class's header file
#include "www_map.h"
#include "www_assert.h"


namespace WeWantWar {


// class constructor
MobileObject::MobileObject() :
  m_position( 0, 0 ),
  m_altitude( 0 ),
  m_verticalSpeed( 0 ),
  m_angle( 0 ),
  m_boundingSphere( 1 ),
  m_collisionFlags( COLLIDE_OBJ | COLLIDE_MAP )
{

}

// class destructor
MobileObject::~MobileObject()
{
	
}


/** Repositions this object.
 */
void MobileObject::position( const eng2d::Vec2D& rP )
{
  m_position = rP;
}



/** Moves this object.
 */
bool MobileObject::move( const eng2d::Vec2D& rMoveVec )
{
  // A flag that tells if there was a collision with map.
  bool c = false;
  m_wallTouch = false;
  
  if ( m_collisionFlags & COLLIDE_MAP ) {
    // Check the collisions
    eng2d::Vec2D tmpPos( m_position );
    tmpPos += rMoveVec;
    
    if ( this->hasMapCollisionAt( tmpPos ) == true ) {
      c = true;
      m_wallTouch = true;
      
      // There is a collision. Try to move only horizontaly
      tmpPos = m_position;
      tmpPos.x( m_position.x() + rMoveVec.x() );
      if ( this->hasMapCollisionAt( tmpPos ) == true ) {
        // Try vertical movement.
        tmpPos = m_position;
        tmpPos.y( m_position.y() + rMoveVec.y() );
        if ( this->hasMapCollisionAt( tmpPos ) == true ) {
          // We can't make the movement.
          tmpPos = m_position;
        }
      }
    }
    m_position = tmpPos;
    
  } else {
    // We don't have to check the collisions
    m_position += rMoveVec;
  }
  return c;
}



/** Sets the angle of this object.
 */
void MobileObject::angle( int a )
{
  m_angle = a & 255;
}



/** Changes the angle by given amount
 */
void MobileObject::changeAngle( int angleChange )
{
  m_angle += angleChange;
  m_angle = m_angle & 255;
}



/** Sets the altitude.
 */
void MobileObject::altitude( float alt )
{
  m_altitude = alt;
}



/** Sets the vertical speed.
 */
void MobileObject::verticalSpeed( float vertSpd )
{
  m_verticalSpeed = vertSpd;
}



/** Sets the radius of the bounding sphere.
 */
void MobileObject::boundingSphere( float r )
{
  WWW_ASSERT( r > 0 );
  m_boundingSphere = r;
}



/** Adds new collision point
 */
void MobileObject::addCollisionPoint( const eng2d::Vec2D& rP )
{
  m_collisionPoints.push_back( rP );
}



/** Modifies an existing collisionpoint.
 */
void MobileObject::setCollisionPoint( int cindex, const eng2d::Vec2D& rP )
{
  WWW_ASSERT( cindex >= 0 );
  WWW_ASSERT( cindex < m_collisionPoints.size() );
  
  m_collisionPoints.at( cindex ) = rP;
}



/** Sets the collisionflags.
 */
void MobileObject::setCollisionFlags( CollFlags flags )
{
  m_collisionFlags = flags;
}



/** Sets the force-vector.
 */
void MobileObject::setForce( const eng2d::Vec2D& rF )
{
  m_force = rF;
}



/** Sets the m_wallTouch - member
 */
void MobileObject::setWallTouch( bool wtouch )
{
  m_wallTouch = wtouch;
}



///
/// GETTER METHODS
/// ==============


/** Returns the current position
 */
eng2d::Vec2D MobileObject::position() const
{
  return m_position;
}



/** Returns the current rotation angle.
 */
int MobileObject::angle() const
{
  return m_angle;
}



/** Returns the altitude.
 */
float MobileObject::altitude() const
{
  return m_altitude;
}



/** Returns the vertical speed.
 */
float MobileObject::verticalSpeed() const
{
  return m_verticalSpeed;
}



/** Returns the bounding sphere.
 */
float MobileObject::boundingSphere() const
{
  return m_boundingSphere;
}



/** Returns the requested collision point.
 */
eng2d::Vec2D MobileObject::getCollisionPoint( int cindex ) const
{
  WWW_ASSERT( cindex >= 0 );
  WWW_ASSERT( cindex < m_collisionPoints.size() );
  return m_collisionPoints.at( cindex );
}



/** Returns the number of collisionpoints this object has.
 */
int MobileObject::getNumOfCollisionPoints() const
{
  return m_collisionPoints.size();
}



/** Tells if this object is colliding with given object.
 */
bool MobileObject::collide( const MobileObject* pObj ) const
{
  // Is the collisionflag set.
  if ( m_collisionFlags & COLLIDE_OBJ ) {
    eng2d::Vec2D distVec( m_position );
    distVec -= pObj->position();
  
    if ( distVec.length() < ( m_boundingSphere + pObj->m_boundingSphere) ) {
      return true;
    }
  }
  return false;
}



/** Tells if this object colliding with given point.
 */
bool MobileObject::collide( const eng2d::Vec2D& rP ) const
{
  eng2d::Vec2D distVec( m_position );
  distVec -= rP;
  if ( distVec.length() < m_boundingSphere ) {
    return true;
  }
  return false;
}



/** Tells if this object collides to the map.
 */
bool MobileObject::hasMapCollision() const
{
  if ( m_collisionFlags & COLLIDE_MAP ) {
    // Check the collisionpoints. The coordinates of the collisionpoints
    // are relative to the object's position so we add our position
    // to them.
    for ( int i=0; i < m_collisionPoints.size(); i++ ) {
      eng2d::Vec2D tmp( m_collisionPoints.at(i) );
      tmp += m_position;
      
      if ( Map::collide( tmp ) == true ) {
        // We're colliding.
        return true;
      }
    }
  }
  return false;
}



/** Tells if this object would collide to the map at given coordinate.
 */
bool MobileObject::hasMapCollisionAt( const eng2d::Vec2D& rP ) const
{
  if ( m_collisionFlags & COLLIDE_MAP ) {
    // Check the collision
    for ( int i=0; i < m_collisionPoints.size(); i++ ) {
      eng2d::Vec2D tmp( m_collisionPoints.at(i) );
      tmp += rP;
      
      if ( Map::collide( tmp ) == true ) {
        // We're colliding
        return true;
      }
    }
  }
  return false;
}



/** Returns the collisionflags.
 */
MobileObject::CollFlags MobileObject::getCollisionFlags() const
{
  return m_collisionFlags;
}



/** Returns the m_wallTouch-member
 */
bool MobileObject::getWallTouch() const
{
  return m_wallTouch;
}



///
/// PROTECTED METHODS
/// =================


/** Updates the movement.
 */
void MobileObject::updateMovement()
{
  // Apply the force-vector.
  if ( m_force.length() > 0.5 ) {
    this->move( m_force );
    m_force.scale( 0.8 );
  }
  
  // Apply the vertical speed.
  m_altitude += m_verticalSpeed;
}


} // end of namespace
